resource "random_id" "uid" {
  byte_length = 4
  keepers = {
    namespace = var.label.namespace
  }
}

module "label" {
  source     = "cloudposse/label/null"
  namespace  = var.label.namespace
  delimiter  = "-"
  attributes = ["${random_id.uid.hex}"]
}

# プライベートネットワーク（スイッチ）
module "label_switch" {
  source  = "cloudposse/label/null"
  name    = "switch"
  context = module.label.context
}

resource "sakuracloud_switch" "switch" {
  name = module.label_switch.id
  tags = module.label_switch.attributes
}

# VPNルータ
module "label_vpn_router" {
  source  = "cloudposse/label/null"
  name    = "vpnr"
  context = module.label.context
}

resource "sakuracloud_vpc_router" "vpn_router" {
  name                = module.label_vpn_router.id
  internet_connection = true
  plan                = "standard"
  tags                = module.label_vpn_router.attributes

  # プライベートネットワークインターフェース
  private_network_interface {
    index        = 1
    switch_id    = sakuracloud_switch.switch.id
    ip_addresses = [var.private_network.gateway_ip]
    netmask      = var.private_network.netmask
  }

  # L2TP/IPsec リモートアクセスVPN設定
  l2tp {
    pre_shared_secret = var.vpn_l2tp_pre_shared_secret
    range_start       = var.vpn_l2tp_range.start
    range_stop        = var.vpn_l2tp_range.stop
  }

  # VPNユーザー設定
  dynamic "user" {
    for_each = var.vpn_users
    content {
      name     = user.value.name
      password = user.value.password
    }
  }

  # ファイアウォール設定（受信方向）
  firewall {
    direction = "receive"

    # IKE (Internet Key Exchange) - UDP 500
    expression {
      protocol            = "udp"
      source_network      = var.vpn_firewall_source_network
      source_port         = ""
      destination_network = ""
      destination_port    = "500"
      allow               = true
      description         = "Allow IKE"
    }

    # IPsec NAT-Traversal - UDP 4500
    expression {
      protocol            = "udp"
      source_network      = var.vpn_firewall_source_network
      source_port         = ""
      destination_network = ""
      destination_port    = "4500"
      allow               = true
      description         = "Allow NAT-T"
    }

    # L2TP - UDP 1701
    expression {
      protocol            = "udp"
      source_network      = var.vpn_firewall_source_network
      source_port         = ""
      destination_network = ""
      destination_port    = "1701"
      allow               = true
      description         = "Allow L2TP"
    }

    # Deny all other traffic
    expression {
      protocol            = "ip"
      source_network      = ""
      source_port         = ""
      destination_network = ""
      destination_port    = ""
      allow               = false
      description         = "Deny all other"
    }
  }
}

module "label_postgres" {
  source  = "cloudposse/label/null"
  name    = "psql"
  context = module.label.context
}

resource "sakuracloud_database" "postgres" {
  name          = module.label_postgres.id
  database_type = "postgres"
  plan          = "10g"
  tags          = module.label_postgres.attributes

  username = var.postgres_user.username
  password = var.postgres_user.password
  network_interface {
    switch_id  = sakuracloud_switch.switch.id
    gateway    = var.private_network.gateway_ip
    ip_address = var.postgres.ip
    netmask    = var.private_network.netmask
  }
}
